/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import javax.ide.extension.ElementVisitor;
import javax.ide.util.Version;
import javax.ide.extension.ExtensionHook;
import java.util.logging.Level;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.Extension;
import java.net.URI;

import java.util.ArrayList;
import java.util.Collection;

import java.util.Collections;

import java.util.logging.Logger;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementVisitorFactory;
import javax.ide.net.VirtualFileSystem;

/**
 * A base class for ExtensionVisitor which processes the attributes of the
 * extension tag. This is also used by a minimal processor in DependencyTree.
 */
public abstract class BaseExtensionVisitor extends ElementVisitor
{
  private static final Logger LOG = Logger.getLogger( BaseExtensionVisitor.class.getName() );
  
  /**
   * The xml element name for the extension element.
   */
  public static final ElementName ELEMENT = 
    new ElementName( ExtensionHook.MANIFEST_XMLNS, "extension" );
      
  /**
   * The xml element name for the classpath element.
   */
  protected static final ElementName CLASSPATH_ELEMENT =
    new ElementName( ExtensionHook.MANIFEST_XMLNS, "classpath" );      
   
  static final ElementName HOOKS =
    new ElementName( ExtensionHook.MANIFEST_XMLNS, "hooks" );
   
  private ElementVisitor _hooksVisitor = createHooksVisitor();
  private ElementVisitorFactory _hookVisitorFactory;
   
  /**
   * The key for the current <tt>ExtensionSource</tt> instance in the scope
   * data map.
   */
  public static final String KEY_EXTENSION_SOURCE = "extSource";   
      
  /**
   * The max ESDK version that this processor will accept. Any
   * version greater than this will result in an error (earlier versions
   * are OK).
   */
  private static final Version MAX_ESDK_VERSION = 
    new Version( "1.0" );

  /**
   * Process an extension.
   * 
   * @param context a processing context.
   * @return a default extension implementation.
   */
  protected DefaultExtension processExtension( ElementStartContext context )
  {
    String id = context.getAttributeValue( "id" );
    if ( id == null || (id = id.trim()) == "" )
    {
      log( context, Level.SEVERE, "Missing required attribute 'id'" );
      return null;
    }
    
    String version = context.getAttributeValue( "version" );
    if ( version == null || ( version = version.trim()) == "" )
    {
      log( context, Level.SEVERE, "Missing required attribute 'version'" );
      return null;
    }
    
    Version parsedVersion;
    try
    {
      parsedVersion = new Version( version );
    }
    catch ( NumberFormatException nfe )
    {
      log( context, Level.SEVERE, "Incorrectly formed version: " + version );
      return null;
    }
    
    
    String esdkversion = context.getAttributeValue( "esdk-version" );
    if ( esdkversion == null || ( esdkversion = esdkversion.trim()) == "" )
    {
      log( context, Level.SEVERE, "Missing required attribute 'esdk-version'" );
      return null;
    }
    
    Version parsedESDKVersion;
    try
    {
      parsedESDKVersion = new Version( esdkversion );
    }
    catch ( NumberFormatException nfe )
    {
      log( context, Level.SEVERE, "Incorrectly formed version: " + esdkversion );
      return null;
    }
    
    if ( parsedESDKVersion.compareTo( MAX_ESDK_VERSION ) > 0 )
    {
      log( context, Level.SEVERE,  
        "ESDK version "+esdkversion+" is too high. Maximum supported version is "+MAX_ESDK_VERSION );
      return null;
    }
    
    DefaultExtension theExtension = new DefaultExtension( id );
    theExtension.setVersion( parsedVersion );
    theExtension.setEDKVersion( parsedESDKVersion );
    theExtension.setSource( (ExtensionSource)
      context.getScopeData().get( ExtensionVisitor.KEY_EXTENSION_SOURCE )
    );
    
    context.getScopeData().put( ExtensionHook.KEY_EXTENSION, theExtension );  
    
    context.registerChildVisitor( HOOKS, _hooksVisitor );    
    
    return theExtension;
  }

  public final void end( ElementEndContext end )
  {
    Extension extension = (Extension) end.getScopeData().get( ExtensionHook.KEY_EXTENSION );
    extension( end, extension );
    addExtensionSourceToClasspath( end );
  }
  
  protected void extension( ElementContext context, Extension extension )
  {
    
  }

  /**
   * Get the extension being processed.
   * 
   * @param context a processing context.
   * @return the extension being processed in this context.
   */
  protected static final DefaultExtension getExtension( ElementContext context )
  {
    return (DefaultExtension) context.getScopeData().get( 
      ExtensionHook.KEY_EXTENSION );
  }
  
  /**
   * Get the source for the extension being processed.
   * 
   * @param context a processing context.
   * @return the extension source for this context.
   */
  protected static final ExtensionSource getSource( ElementContext context )
  {
    return (ExtensionSource) context.getScopeData().get(
      KEY_EXTENSION_SOURCE );
  }  

  /**
   * Add the specified entry to the classpath of the class loader which is
   * loading this extension and its dependent classes.
   * 
   * @param context the current processing context.
   * @param ext the extension being processed. This may be in a partially
   *    initialized state.
   * @param entry a classpath entry used by the current extension.
   */
  protected abstract void addToClasspath( ElementContext context, 
    Extension ext, URI entry );
    
  /**
   * Sets the hook visitor factory to use. 
   * 
   * @param hookFactory the hook visitor factory to use.
   * @since 2.0
   */
  protected final void setHookVisitorFactory( ElementVisitorFactory hookFactory )
  {
    _hookVisitorFactory = hookFactory;
  }
  
  protected ElementVisitor createClasspathVisitor()
  {
    return new ClasspathVisitor();
  }
    
  /**
   * A visitor for the classpaths element. This is not registered by default
   * by this implementation, however the implementation class is available
   * so that subclasses can easily register it.
   */
  public final class ClasspathsVisitor extends ElementVisitor
  {
    private final ElementVisitor _cpVisitor = createClasspathVisitor();
  
    public void start( ElementStartContext context )
    {
      context.registerChildVisitor( CLASSPATH_ELEMENT, _cpVisitor );
    }
  }
  
  /**
   * Match the specified wildcard expression in the specified parent directory.
   * 
   * @param parent a directory. Guaranteed to exist and be a directory.
   * @param expression a wildcard expression. Currently only supports *
   *    wildcard expansion, and not ?.
   * @return a collection of matching URIs. Can be an empty collection or
   *    null (signifies that the expression was invalid).
   *    
   * @since 2.0
   */
  public static final Collection<URI> matchWildcard( URI parent, String expression )
  {
    URI[] children = VirtualFileSystem.getVirtualFileSystem().list( parent );
    
    // Extra sanity checking. Null if dir does not exist. Shouldn't happen
    // if caller honors javadoc contract, but play safe...
    if ( children == null ) return null;      
    
    Collection<URI> matching = new ArrayList<URI>();
    SimpleGlobMatcher matcher = new SimpleGlobMatcher( expression );
    for ( URI uri : children )
    {
      String fileName = VirtualFileSystem.getVirtualFileSystem().getFileName( uri );
      if ( fileName == null ) continue;
      if ( !matcher.matches( fileName ) ) continue;
      
      matching.add( uri );
    }
    
    LOG.fine( "Wildcard " + expression + " expanded to: " + matching );
    
    return matching;
  }
  
  /**
   * A visitor for the classpath element.
   */
  public class ClasspathVisitor extends ElementVisitor
  {
  
    /**
     * Gets a single classpath entry from the specified context.
     * 
     * @param context a processing context.
     * @return a single classpath entry from the context, or null if no
     *    entry is available.
     */
    protected URI getClasspathEntry( ElementEndContext context )
    {
      String text = context.getText().trim();
      if ( text.length() == 0 )
      {
        log( context, Level.WARNING,
          "Empty classpath definition"
        );
        return null;
      }

      return resolveURI( context, text );
    }
    
    private URI resolveURI( ElementContext context, String text )
    {
      return getSource( context ).resolvePath( getExtension( context ), text );
    }
    
    /**
     * Validate a classpath entry. If the classpath entry is illegal and 
     * should not be added to the classpath, return <tt>false</tt>.<p>
     * 
     * This implementation unconditionally returns <tt>true</tt>.
     * 
     * @param context the current XML processing context.
     * @param uri a URI for a classpath entry. Not null.
     * @return <tt>true</tt> if the classpath entry should be added, 
     *    <tt>false</tt> otherwise.
     * 
     * @since 2.0
     */
    protected boolean acceptClasspathEntry( ElementContext context, URI uri )
    {
      return true;
    }

    /**
     * Get multiple classpath entries from a single context. This method
     * supports the use of wildcards.
     * 
     * Note: this method intentionally only handles the case where the 
     * extension path contains a wildcard in order to maintain compatibility
     * with 1.0. The getClasspathEntry( ElementEndContext ) must be called
     * if a <classpath> element contains a reference to only one
     * classpath entry (i.e. contains no wildcards).
     * 
     * @since 2.0
     * 
     * @return null if the specified context contains no wildcards.
     */
    private Collection<URI> getClasspathEntries( ElementEndContext context )
    {
      String text = context.getText();
      if ( text == null ) return null;
      
      text = text.trim();
      if ( text.length() == 0 ) return null;
      
      // We support wildcards *only* on the last path element. 
      int posOfWildcard = -1;
      int posOfSeparator = -1;
      for ( int i=text.length()-1; i >= 0; i-- )
      {
        char c = text.charAt( i );
        if ( c == '*' ) posOfWildcard = i;
        if ( c == '/' ) 
        {
          posOfSeparator = i;
          break;
        }
      }
      
      // If we have no wildcard, or no separator, or the separator is the last
      // character of the path, then we have nothing expandable.
      if ( posOfWildcard == -1 || posOfSeparator == -1 ||
           posOfSeparator == text.length() - 1 ) return null;
      
      LOG.fine( "Procesing wildcard classpath entry: " + text );
      
      String dirPath = text.substring( 0, posOfSeparator );
      String glob = text.substring( posOfSeparator + 1 );
      URI parentURI = resolveURI( context, dirPath );
      
      // The parent must exist, else we can do no wildcard matching on it.
      if ( !VirtualFileSystem.getVirtualFileSystem().isDirectory( parentURI ) ) {
        LOG.fine( "Parent directory for wildcard entry does not exist: " + parentURI );
        return null;
      }
      
      return matchWildcard( parentURI, glob );
    }
  
    public final void end( ElementEndContext context )
    {
      Collection<URI> allURIs = getClasspathEntries( context );
      
      if ( allURIs == null ) // entry did not contain a wildcard.
      {
        // As noted above, we don't just handle everything in
        // getClasspathEntries for compatibility.
        URI singleEntry = getClasspathEntry( context );
        if ( singleEntry == null ) allURIs = Collections.emptyList();
        else allURIs = Collections.singleton( singleEntry );
      }

      for ( URI uri : allURIs )
      {
        addURIToClasspath( context, getExtension( context ), uri );
      }
    }
    
    private void addURIToClasspath( ElementContext context, Extension ext, 
      URI entry )
    {
      if ( !acceptClasspathEntry( context, entry ) ) return;
      
      if ( ext instanceof DefaultExtension )
        ((DefaultExtension)ext).getClassPath().add( entry );
      
      addToClasspath( context, ext, entry );
    }
  }    
  
  protected ElementVisitor createHooksVisitor()
  {
    return new HooksVisitor();
  }  
  
  private final class HooksVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      if ( _hookVisitorFactory != null )
      {
        context.registerVisitorFactory( _hookVisitorFactory );
      }
      addExtensionSourceToClasspath( context );
    }
  }  
  
  /**
   * Called as late as possible to add the extension source to the classpath.
   * In this implementation, this method is called when the &lt;hooks> element
   * is encountered, or the end of the extension is reached, whichever comes
   * first. This method may be called multiple times for the same extension.
   * 
   * @param context
   */
  protected void addExtensionSourceToClasspath( ElementContext context ) 
  {
    DefaultExtension extension = getExtension( context );
    ExtensionSource source = extension.getSource();
    
    LOG.fine( "Adding source for " + extension.getID() + " to classpath: " + source.getClasspathEntry() );
    addToClasspath( context, extension, source.getClasspathEntry() );
  }
}
