/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.util.ArrayList;
import java.util.Collection;

import java.util.Collections;
import java.util.HashSet;

import javax.ide.Identifiable;
import javax.ide.extension.Extension;
import javax.ide.extension.ExtensionDependency;
import javax.ide.extension.ExtensionRegistry;
import javax.ide.extension.PlatformInfo;
import javax.ide.net.URIPath;
import javax.ide.util.Version;
 
/**
 * This class maintains general information about an extension. Information
 * gathered from the extension manifest is stored in this class.
 * Such information include the extension's id, owner, version information, 
 * etc..
 * <p>
 * IDE providers can use this class to to record the extension information 
 * section of the extension manifest.
 */

public final class DefaultExtension implements Extension, Identifiable, Comparable
{
  private String              _id;
  private String              _name;
  private String              _rawName;
  private String              _rawOwner;
  private Version             _version;
  private Version             _edkVersion;
  private String              _owner;
  private Collection           _dependencies = new ArrayList();
  private URIPath             _classpath = new URIPath();
  private PlatformInfo        _platformInfo;
  private ExtensionSource     _source;
  private boolean             _cycleDetector = false;
  private Collection<String> _allImports;

  //---------------------------------------------------------------------------
  // Public methods.
  //---------------------------------------------------------------------------

  /**
   * Default constructor.
   */
  public DefaultExtension()
  {
  }
  
  public DefaultExtension( String id )
  {
    _id = id;
  }

  public void setSource( ExtensionSource source )
  {
    _source = source;
  }
  
  public ExtensionSource getSource()
  {
    return _source;
  }

  public String getID()
  {
    return _id;
  }

  /*-
   * Extension method.
   */
  public String getName()
  {
    return _name;
  }

  /*-
   * Extension method.
   */
  public String getOwner()
  {
    return _owner;
  }

  /*-
   * Extension method.
   */
  public Version getVersion()
  {
    return _version;
  }

  /*-
   * Extension method.
   */
  public Version getEDKVersion() 
  {
    return _edkVersion;  
  }

  /*-
   * Extension interface method.
   */
  public PlatformInfo getPlatformInfo()
  {
    return _platformInfo;
  }

  /*-
   * Extension method.
   */
  public Collection<ExtensionDependency> getDependencies()
  {
    return _dependencies;
  }
  
  /**
   * Gets all imported extensions for this extension. This walks up the entire
   * dependency graph for this extension, collecting all imported extensions.
   * It will stop if it detects a cycle, in which case, the collection of 
   * dependencies may not be complete. <p>
   * 
   * It must be able to retrieve extensions from the specified registry. So
   * for any given extension id x in the set of dependencies for this
   * extension, ExtensionRegistry.findExtension( x ) must not throw 
   * exceptions. If the method returns null for a given extension id, then
   * that dependency will not be included.<p>
   * 
   * The dependencies are not guaranteed to be in any particular order.
   * 
   * @return an immutable collection of all imported extensions for this
   *    extension.
   */
  public Collection<String> getAllImportedExtensions( ExtensionRegistry registry )
  {
    if ( _allImports == null )
    {
      Collection<String> allDependencies = new HashSet<String>();
      getAllImportedExtensionsImpl( allDependencies, registry );
      _allImports = Collections.unmodifiableCollection( allDependencies );
    }

    return _allImports;
  }
  
  private void getAllImportedExtensionsImpl( Collection<String> deps, 
    ExtensionRegistry registry )
  {
    if ( _cycleDetector )
    {
      return;
    }
    try
    {
      _cycleDetector = true;
      for ( ExtensionDependency dep : getDependencies() )
      {
        final String id = dep.getID();
        Extension ext = registry.findExtension( id );
        if ( ext != null && ext instanceof DefaultExtension )
        {
          deps.add( id );
          ((DefaultExtension)ext).getAllImportedExtensionsImpl( deps, registry );
        }
      }
    }
    finally
    {
      _cycleDetector = false;
    }
  }

  /*-
   * Extension method.
   */
  public URIPath getClassPath()
  {
    return _classpath;
  }

  String getRawName()
  {
    return _rawName;
  }
  
  void setRawName( String rawName )
  {
    _rawName = rawName == null ? "" : rawName.trim();
  }
  
  String getRawOwner()
  {
    return _rawOwner;
  }
  
  void setRawOwner( String rawOwner )
  {
    _rawOwner =rawOwner;
  }

  //---------------------------------------------------------------------------
  // Methods used by IDE providers to set property values defined in the 
  // extension manifest.
  //---------------------------------------------------------------------------
   
  /**
   * Set the name of this extension. The name should not include
   * the extension version label.
   *
   * @param name short name for this extension.
   */
  public void setName( String name )
  {
    _name = name;
  }

  /**
   * Set the extension owner name.
   *
   * @param owner label identifying the extension writer.
   */
  public void setOwner( String owner )
  {
    _owner = owner;
  }

  /**
   * Set the extension version information.
   *
   * @param version the extension version specification. 
   */
  public void setVersion( Version version )
  {
    _version = version;
  }

  /**
   * Set the EDK version.
   *
   * @param edkVersion The version of the EDK used to implement this extension.
   */
  public void setEDKVersion( Version edkVersion ) 
  {
    _edkVersion = edkVersion;  
  }

  /**
   * Set the platform specific info required for this extension to work.
   *
   * @param info The {@link PlatformInfo}.
   */
  public void setPlatformInfo( PlatformInfo info )
  {
    _platformInfo = info;
  }

  /**
   * Set a list of {@link Extension} identifiers this extension depends on.
   *
   * @param dependency Dependencies list. 
   */
  public void addDependency( ExtensionDependency dependency )
  {
    _dependencies.add( dependency );
  }

  /**
   * Set an additional classpath where classes used by this extension 
   * can be found.
   *
   * @param classpath Classpath where classes used by this extension are
   * found.
   */
  public void setClassPath( URIPath classpath )
  {
    _classpath = classpath;
  }
  
  public String toString()
  {
    return getID() + " " + String.valueOf( getVersion() );
  }
  
  /**
   * Compare two extensions on their names.
   * 
   * @param o the extension to compare.
   * @return the value 0 if the argument string is equal to this string; 
   * a value less than 0 if this string is lexicographically less than the 
   * string argument; and a value greater than 0 if this string is 
   * lexicographically greater than the string argument
   */
  public int compareTo(Object o)
  {
    if ( o instanceof Extension )
    {
      return getName().compareTo( ((Extension)o).getName() );
    }
    return -1;
  }
}
