package javax.ide.extension.spi;

import java.util.logging.Level;
import java.util.logging.LogRecord;

import javax.ide.extension.ElementContext;

import javax.ide.extension.ElementVisitor;

import javax.ide.extension.Extension;

import org.xml.sax.Locator;

/**
 * Extension log records hold positional information about the location where
 * log messages occurred while processing the extension manifest.
 */
public final class ExtensionLogRecord extends LogRecord
{
  private final Locator _locator;

  public ExtensionLogRecord( Locator locator, Level level, String msg )
  {
    super( level, msg );
    _locator = copyLocator( locator );

    // For backwards compatibility (early versions of the API used the
    // parameters array to store the locator)
    if ( _locator != null )
    {
      setParameters( new Object[] { _locator } );
    }
  }

  /**
   * Creates a log record.
   * 
   * @param context the current procesing context.
   * @param level the message log level.
   * @param msg the message to log.
   */
  public ExtensionLogRecord( ElementContext context, Level level, String msg )
  {
    this( (Locator) context.getScopeData().get( ElementVisitor.KEY_LOCATOR ), 
      level, msg );
  }
  
  /**
   * Creates a log record. 
   * 
   * @param level the message log level
   * @param msg the message
   * @param extension the extension 
   * @param lineNumber the line number within the extension manifest.
   * @since 2.0
   */
  public ExtensionLogRecord( Level level, String msg, Extension extension, 
    int lineNumber )
  {
    this( extensionToLocator( extension, lineNumber ), level, msg );
  }
  
  /**
   * Creates a log record. 
   * 
   * @param level the message log level
   * @param msg the message
   * @param extension the extension 
   * @param lineNumber the line number within the extension manifest.
   * @param thrown the exception that was thrown.
   * @since 2.0
   */
  public ExtensionLogRecord( Level level, String msg, Extension extension, 
    int lineNumber, Throwable thrown )
  {
    this( extensionToLocator( extension, lineNumber ), level, msg );
    setThrown( thrown );
  }  
  
  private static Locator copyLocator( Locator locator )
  {
    final int lineNumber = locator.getLineNumber();
    final int colNumber = locator.getColumnNumber();
    final String publicId = locator.getPublicId();
    final String systemId = locator.getSystemId();
    
    return new Locator() 
    {
      public String getPublicId()
      {
        return publicId;
      }

      public String getSystemId()
      {
        return systemId;
      }

      public int getLineNumber()
      {
        return lineNumber;
      }

      public int getColumnNumber()
      {
        return colNumber;
      }
    };
  }
  
  private static Locator extensionToLocator( final Extension extension, 
    final int lineNumber )
  {
    return new Locator() 
    {

      public String getPublicId()
      {
        return null;
      }

      public String getSystemId()
      {
        if ( extension instanceof DefaultExtension )
        {
          ExtensionSource source =  ((DefaultExtension)extension).getSource();
          if ( source == null ) return "";
          return String.valueOf( source.getManifestURI() );
        }
        return "";
      }

      public int getLineNumber()
      {
        return lineNumber;
      }

      public int getColumnNumber()
      {
        return 0;
      }
    };
  }
  
  /**
   * Gets the locator.
   * 
   * @return the locator.
   */
  public Locator getLocator()
  {
    return _locator;
  }
}
