/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.io.IOException;

import java.util.logging.Level;
import java.util.logging.LogRecord;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementVisitor;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;


/**
 * An extensible SAX parser.
 *
 * @author Brian.Duff@oracle.com
 */
public final class SAXManifestParser
{
  private final DefaultElementContext _context;
  private SAXParser _parser = null;
  private Handler _handler = null;
  private Locator _locator;
  
  public SAXManifestParser( DefaultElementContext initialContext )
  {
    _context = initialContext;
  }
  
  /**
   * Get the context
   */
  public ElementContext getContext()
  {
    return _context;
  }
  
  public void parse( InputSource inputSource )
    throws ParserConfigurationException, SAXException, IOException
  {
    parse(null, inputSource);
  }
  
  public void parse( XMLReader reader, InputSource inputSource )
    throws ParserConfigurationException, SAXException, IOException
  {
    if (reader == null)
    {
      if ( _parser == null )
      {
        SAXParserFactory factory = SAXParserFactory.newInstance();
        
        // make namespace aware, since the JAXP default is non-namespace aware
        factory.setNamespaceAware(true);
        _parser = factory.newSAXParser();
      }

      reader = _parser.getXMLReader();
    }
    
    if (_handler == null)
    {
      _handler = new Handler();      
    }
    
    reader.setContentHandler( _handler );
    reader.setErrorHandler( _handler );

    try
    {
      reader.parse( inputSource );       
    }
    finally
    {
      // Always reset the context, otherwise we're potentially stranded
      // with unbalanced start / ends. This fixes bug 5457726.
      _context.reset();    
    }
  }

    
  private class Handler extends DefaultHandler
  {
  
    public void startElement( String uri, String name, String qName, 
      Attributes attributes ) throws SAXException
    {
      ElementVisitor visitor = _context.getVisitorForStartElement( new ElementName( uri, name ) );

      _context.beginElement( uri, name, attributes );
    
      if ( visitor != null )
      {
        recordPosition( _context );
        try
        {
          visitor.start( _context );
        }
        catch ( Throwable re )
        {
          if ( re instanceof ThreadDeath  ) throw (ThreadDeath) re;
          LogRecord r = new ExtensionLogRecord( _locator, Level.SEVERE, "Exception processing manifest: " +
              re.getClass().getName() + ":" + re.getMessage() );
          r.setThrown( re );
          _context.getLogger().log( r );
        }
      }
      
      _context.postBeginElement();
    }

    public void endElement( String uri, String name, String qName )
      throws SAXException
    {
      _context.endElement( uri, name );
      ElementVisitor visitor = _context.getVisitorForEndElement();
      if ( visitor != null )
      {
        recordPosition( _context );
        try
        {
          visitor.end(_context);
        }
        catch ( Throwable re )
        {
          if ( re instanceof ThreadDeath  ) throw (ThreadDeath) re;
          LogRecord r =
            new ExtensionLogRecord( _locator, Level.SEVERE, "Exception processing manifest: " +
              re.getClass().getName() + ":" + re.getMessage() );
          r.setThrown( re );
          _context.getLogger().log( r );
        }
      }
      _context.postEndElement();
    }

    public void characters ( char ch[], int start, int length )
    {
      _context.appendCharacters( ch, start, length );
    }  
    
    public void setDocumentLocator(Locator locator)
    {
      _locator = locator;
    }
    
    private void recordPosition( ElementContext context )
    {    
      context.getScopeData().put( ElementVisitor.KEY_LOCATOR, _locator );
    }
  }
}
