/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.util.AbstractMap;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

/**
 * A map that provides support for scoping.
 */
public final class ScopedMap extends AbstractMap
{
  private final Stack _mapStack = new Stack();
  
  public ScopedMap()
  {
    enterScope();    // Always have a root scope to avoid illegal object states  
  }
  
  void reset()
  {
    while ( _mapStack.size() > 1 )
    {
      _mapStack.pop();
    }
  }
  
  /**
   * Enter a new scope level.  
   */
  public void enterScope()
  {
    _mapStack.push( new HashMap() );
  }
  
  /**
   * Exit the current scope level. Name / value pairs that
   * were defined in the current scope will no longer be available.
   */
  public void exitScope()
  {
    _mapStack.pop();
  }
  
  
  public Object put( Object key, Object value )
  {
    Map topMap = (Map) _mapStack.peek();
    return topMap.put( key, value );
  }
  
  public Object get( Object key )
  {
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      Map thisMap = (Map) i.next();
      Object o = thisMap.get( key );
      if ( o != null )
      {
        return o;
      }
    }
    return null;
  }
  
  public Object remove( Object key )
  {
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      Map thisMap = (Map) i.next(); 
      Object o = thisMap.remove( key );
      if ( o != null )
      {
        return o;
      }
    }
    return null;
  }
  
  public void clear()
  {
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      ((Map)i.next()).clear();
    }
  }
  
  public Set entrySet()
  {
    HashSet hs = new HashSet();
    for ( Iterator i = _mapStack.iterator(); i.hasNext(); )
    {
      Map thisMap = (Map) i.next();
      hs.addAll( thisMap.entrySet() );
    }
    return hs;
  }
}
