/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu.spi;

import javax.ide.extension.MetaClassVisitor;
import javax.ide.IDEConstants;
import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import javax.ide.util.MetaClass;

/**
 * A <code>MenuHook</code> stores the extension manifest 
 * information pertaining to actions, menus, and toolbars.
 */

public final class MenuHook extends ExtensionHook
{
  public static final ElementName ELEMENT = 
    new ElementName( MANIFEST_XMLNS, "menu-hook" );
    
  private static final ElementName ACTIONS = new ElementName(
    MANIFEST_XMLNS, "actions" );
    
  private static final ElementName MENUS = new ElementName(
    MANIFEST_XMLNS, "menus" );
    
  private static final ElementName TOOLBARS = new ElementName(
    MANIFEST_XMLNS, "toolbars" );
    
  private static final ElementName ITEM_FACTORIES = new ElementName(
    MANIFEST_XMLNS, "item-factories" );
  private static final ElementName ITEM_FACTORY_CLASS = new ElementName(
    MANIFEST_XMLNS, "item-factory-class" );
    
  static final String KEY_MENU_MODEL = "menuModel";
  static final String KEY_SECTION_CONTAINER = "sectionContainer";
  
  
  private ElementVisitor _actionsVisitor = new ActionsVisitor();
  private ElementVisitor _toolbarsVisitor = new ToolbarsVisitor();
  private ElementVisitor _menusVisitor = new MenusVisitor();
  private ElementVisitor _itemFactoriesVisitor = new ItemFactoriesVisitor();
  
  
  private final MenuModel _menuModel = createInitialMenuModel();
  
  
  public void start( ElementStartContext context )
  {
    context.getScopeData().put( KEY_MENU_MODEL, _menuModel );
    
    context.registerChildVisitor( TOOLBARS, _toolbarsVisitor );
    context.registerChildVisitor( ACTIONS, _actionsVisitor );
    context.registerChildVisitor( MENUS, _menusVisitor );
    context.registerChildVisitor( ITEM_FACTORIES, _itemFactoriesVisitor );
  }
  
  public MenuModel getModel()
  {
    return _menuModel;
  }

  /**
   * Create the initial menu model. This prepopulates the menu model with the
   * standard JSR-198 items and sections so that other items can be positioned
   * in relation to them.
   * 
   * IDEs may override this to change the position of the standard items based
   * on the layout of their own menus and toolbars.
   * 
   * @return a pre populated initial menu model.
   */
  protected MenuModel createInitialMenuModel()
  {

    MenuModel model = new MenuModel();
    MenuBar mbMain = 
      model.findOrCreatePullDownMenu( IDEConstants.MAIN_WINDOW_MENUBAR_ID ); 
    
    Menu menu = new Menu( IDEConstants.FILE_MENU_ID );
    
    Section section = new Section( IDEConstants.NEW_SECTION_ID );
    menu.addSection( section );
    
    section = new Section( IDEConstants.OPEN_SECTION_ID );
    Item item = new Item( IDEConstants.OPEN_ACTION_ID );
    section.addItem( item );
    menu.addSection( section );
    
    section = new Section( IDEConstants.CLOSE_SECTION_ID );
    menu.addSection( section );
    
    section = new Section( IDEConstants.PRINT_SECTION_ID );
    menu.addSection( section );
    
    mbMain.addMenu( menu );
    
    menu = new Menu( IDEConstants.EDIT_MENU_ID );
    section = new Section( IDEConstants.COPY_PASTE_SECTION_ID );
    section.addItem( new Item( IDEConstants.CUT_ACTION_ID ) );
    section.addItem( new Item( IDEConstants.COPY_ACTION_ID ) );
    section.addItem( new Item( IDEConstants.PASTE_ACTION_ID ) );
    menu.addSection( section );
    mbMain.addMenu( menu );
    
    menu = new Menu( IDEConstants.VIEW_MENU_ID );
    mbMain.addMenu( menu );
    
    menu = new Menu( IDEConstants.HELP_MENU_ID );
    mbMain.addMenu( menu );
    

    return model;
  }
  
  private class ItemFactoriesVisitor extends ElementVisitor
  {
    private final ElementVisitor _itemFactoryClassVisitor = new ItemFactoryClassVisitor();
    public void start( ElementStartContext context )
    {
      context.registerChildVisitor( ITEM_FACTORY_CLASS, _itemFactoryClassVisitor );
    }
  }
  
  private class ItemFactoryClassVisitor extends MetaClassVisitor
  {
    protected void metaClass(ElementContext context, MetaClass mc)
    {
      getModel().addItemFactory( mc );
    }
  }
  
  

}
