/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu.spi;

import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;
import java.util.logging.Level;

/**
 * Visits the JSR-198 menu hook element.
 */
final class MenusVisitor extends ElementVisitor
{
  private static final ElementName MENUBAR = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "menubar" );    
  private static final ElementName POPUP = new ElementName( 
    ExtensionHook.MANIFEST_XMLNS, "popup" );

  static final String KEY_MENU_BAR = "menuBar";

  private ElementVisitor _menubarVisitor = new MenuBarVisitor();
  private ElementVisitor _menuVisitor = new MenuVisitor();
  private ElementVisitor _popupVisitor = new PopupVisitor();
  private ElementVisitor _sectionVisitor = new SectionVisitor();
  
  /**
   * Process the start of an XML element.
   * 
   * @param context the XML processing context.
   */
  public void start( ElementStartContext context )
  {
    context.registerChildVisitor( MENUBAR, _menubarVisitor );
    context.registerChildVisitor( POPUP, _popupVisitor );
  }
  
  private final class MenuBarVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      String id = context.getAttributeValue( "id" );
      if ( id == null || ( id = id.trim() ) == "" )
      {
        log( context, Level.SEVERE, "Required attribute 'id' missing" );
        return;
      }
      MenuModel model = 
        (MenuModel) context.getScopeData().get( MenuHook.KEY_MENU_MODEL );
      MenuBar menuBar = model.findOrCreatePullDownMenu( id );
      context.getScopeData().put( KEY_MENU_BAR, menuBar );
      
      context.registerChildVisitor( MenuVisitor.MENU, _menuVisitor );
    }
  }
  
  private final class PopupVisitor extends ElementVisitor
  {
    public void start( ElementStartContext context )
    {
      String id = context.getAttributeValue( "id" );
      if ( id == null || ( id = id.trim() ) == "" )
      {
        log( context, Level.SEVERE, "Required attribute 'id' missing" );
        return;
      }
      MenuModel model = 
        (MenuModel) context.getScopeData().get( MenuHook.KEY_MENU_MODEL );
      PopupMenu popup = model.findOrCreatePopupMenu( id );
      context.getScopeData().put( MenuHook.KEY_SECTION_CONTAINER, popup );
      
      context.registerChildVisitor( SectionVisitor.SECTION, _sectionVisitor );
    }
  }







}
