/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.menu.spi;

import java.util.logging.Level;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ExtensionHook;

/**
 * Visitor for menu sections.
 */
final class SectionVisitor extends PositionableVisitor
{
  
  final static ElementName SECTION = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "section" );

  private final static ElementName ITEM = new ElementName(
    ExtensionHook.MANIFEST_XMLNS, "item" );
    
  
  private ElementVisitor _menuVisitor = new MenuInSectionVisitor();
  private ElementVisitor _itemVisitor = new ItemVisitor();

  private final static String KEY_SECTION = "section";

  protected void positionable(ElementStartContext context, String id, String before, 
    String after)
  {
    SectionContainer container = (SectionContainer) 
      context.getScopeData().get( MenuHook.KEY_SECTION_CONTAINER );
    
    Section section = container.getSection( id );
    if ( section == null )
    {
      section = new Section( id );
      container.addSection( section );      
    }

    if ( before != null  )
    {
      section.setBefore( before );
    }
    else if ( after != null )
    {
      section.setAfter( after );
    }
    
    
    context.getScopeData().put( KEY_SECTION, section );
    
    context.registerChildVisitor( MenuVisitor.MENU, _menuVisitor );
    context.registerChildVisitor( ITEM, _itemVisitor );
  }
  
  private Section getSection( ElementContext context )
  {
    return (Section) context.getScopeData().get( KEY_SECTION );
  }
  
  
  private final class ItemVisitor extends PositionableVisitor
  {
    ItemVisitor()
    {
      super( "action-ref" );    // items use action-ref instead of id.
    }
  
    protected void positionable(ElementStartContext context, String id, 
      String before, String after)
    {
      Section parentSection = getSection( context );
      Positionable pos = parentSection.getItem( id );
      
      if ( pos == null )
      {
        pos = new Item( id );
        parentSection.addItem( pos );
      }
      else
      {
        if ( pos instanceof Menu )
        {
          log( context, Level.SEVERE, 
            "'"+id+"' is already defined as a menu in section '"+parentSection.getID()+"'" );
          return;          
        }
      }
      
      Item item = (Item) pos;
      
      String radioGroupId = context.getAttributeValue( "radiogroup-id" );
      if ( radioGroupId != null )
      {
        radioGroupId = radioGroupId.trim();
        item.setRadioGroupID( radioGroupId );
      }
      
      if ( before != null ) item.setBefore( before );
      else if ( after != null ) item.setAfter( after );
    }
  }
  
  private final class MenuInSectionVisitor extends MenuVisitor
  {
    protected void positionable(ElementStartContext context, String id,
      String before, String after)
    {
      Section parentSection = getSection( context );
      
      Positionable pos = parentSection.getItem( id );

      if ( pos == null )
      {
        pos = new Menu( id );
        parentSection.addItem( pos );
      }
      else
      {
        if ( pos instanceof Item )
        {
          log( context, Level.SEVERE, 
            "'"+id+"' is already defined as an item in section '"+parentSection.getID()+"'" );
          return;
        }
      }
            
      final Menu menu = (Menu) pos;
      processMenu( context, menu, before, after );
      
    }    
  }

}
