/*
 * @(#)TypeD.java
 */

package javax.ide.model.java.declaration;

import java.util.Collection;

/**
 * Represents a type.
 *
 * <h3> Members </h3>
 *
 * The order of the member listing should be determined by the type
 * hierarchy. If type B occurs before type A in the type hierarchy,
 * then all of B's declared members should occur before any of A's
 * declared members in the member collection. Consider the following
 * two classes. <p/>
 *
 * <pre>
 *   class A
 *   {
 *     int a;
 *   }
 *   class B extends A
 *   {
 *     int b;
 *   }
 * </pre>
 *
 * The iterator for the collection returned for the member fields of B
 * should always return <code>B.b</code> before <code>A.a</code>.
 * Order within the same declaring type is not defined. <p/>
 *
 * @author Andy Yu
 */
public interface TypeD
  extends Declaration, HasTypeD, HasNameD
{
  public static final TypeD[] EMPTY_ARRAY = new TypeD[ 0 ];

  /**
   * Gets itself as its type.
   *
   * @return The type of a type declaration is trivially itself.
   */
  public TypeD getType();

  /**
   * Gets the type erasure of this type.
   *
   * @return The type declaration representing the type erasure of
   * this type.
   */
  public ClassD getTypeErasure();

  /**
   * True if this is a primitive type, including the special
   * <code>void</code> type. <p/>
   *
   * @return True if this type is a primitive type.
   */
  public boolean isPrimitive();

  /**
   * True if this is an array type. <p/>
   *
   * @return True if this type is an array type.
   */
  public boolean isArray();


  // ----------------------------------------------------------------------

  /**
   * True if this is an enum type. <p/>
   *
   * @return True if this an enumeration type. False otherwise.
   */
  public boolean isEnum();

  /**
   * True if this is an annotation type. <p/>
   *
   * @return True if this is an annotation type. False otherwise.
   */
  public boolean isAnnotation();

  /**
   * True if this is an interface. This includes "interface" types and
   * "annotation" types. <p/>
   *
   * @return True if this is an interface type.
   */
  public boolean isInterface();


  // ----------------------------------------------------------------------

  /**
   * Gets the fully qualified name of this type in source format.
   * Here, "source format" means the format of a type reference in a
   * source compilation unit. A type's source name uniquely identifies
   * a type in a specific scope but not in the general scope. <p/>
   *
   * An array type returns the fully qualified name of its component
   * type suffixed by the appropriate number of brackets.
   * Parameterized types return the fully qualified name of its
   * unparameterize type suffixed with the type parameters. A local
   * inner class returns only its simple name because that is how it
   * is referred to in a source compilation unit. <p/>
   *
   * Note: Anonymous inner classes return the empty string because
   * they may not be referred to in a source compilation unit. <p/>
   *
   * @return The fully qualified source name for this type, empty for
   * anonymous classes. Examples: <code>java.lang.Object</code>,
   * <code>java.util.Map&lt;String,String&gt;</code>,
   * <code>java.lang.String[][]</code>
   */
  public String getQualifiedName();

  /**
   * Gets the descriptor for this type, as defined by the JVMS2. <p/>
   *
   * For jdk < 1.5, the "FieldDescriptor" is returned (JVMS2). For
   * jdk >= 1.5, a type signature of the type erasure is returned. <p/>
   *
   * Note: The descriptor is equivalent to the type signature of this
   * type's erasure. <p/>
   *
   * @return The descriptor as defined in the JVMS2. Always non-null.
   * e.g. "I", "Ljava/lang/Object;"
   */
  public String getDescriptor();

  /**
   * Gets the type signature for this type, as defined by the
   * JVMS3. <p/>
   *
   * A ClassD returns its "ClassTypeSignature" (JVMS3). A ClassD is
   * uniquely identified by its type signature. Example value:
   * <code>Ljava/util/List<Ljava/lang/Object;>;</code> <p/>
   *
   * A TypeVariableD returns its "TypeVariableSignature" (JVMS3). A
   * TypeVariableD is uniquely identified by its owning declaration
   * and its type signature. Example value: <code>TE;</code>. <p/>
   *
   * A WildcardTypeD returns its "TypeArgument" value (JVMS3). A
   * WildcardTypeD is uniquely identified by its type signature.
   * Example values: <code>*</code> <code>+Ljava/lang/Object;</code>
   * <p/>
   *
   * @return The type signature as defined in the JVMS3. Always
   * non-null.
   */
  public String getTypeSignature();


  // ----------------------------------------------------------------------

  /**
   * Gets this type's superclass. The name "getSuperclass" was chosen
   * to be consistent with java/lang/Class.
   *
   * @return The type declaration for this type's superclass. Behavior
   * should "declaration" reflection. <p/>
   *
   * If this is a ClassD: Null is returned for java/lang/Object and
   * primitive types, non-null otherwise. Where no base class was
   * declared, java/lang/Object is returned, even for interfaces. <p/>
   *
   * If this is a TypeVariableD: Null if no superclass was declared
   * AND an interface as declared. <p/>
   *
   * If this is a WildcardTypeD: Always non-null. If no superclass was
   * declared, then the TypeD for java/lang/Object is returned. <p/>
   *
   * Note: Wildcard types may have a type variable as an upper bound,
   * therefore, this must return a TypeD instead of a ClassD. <p/>
   */
  public TypeD getSuperclass();

  /**
   * Gets this type's super-interfaces. The name "getInterfaces" was
   * chosen to be consistent with java/lang/Class.
   *
   * @return The collection of type declarations for this type's
   * super-interfaces. May be zero-length. Will include all implicit
   * base interfaces, e.g. all array types implement
   * java/lang/Cloneable. <p/>
   *
   * Collection of TypeDs.
   */
  public Collection getInterfaces();

  /**
   * Recursively gets all direct supertypes. <p/>
   *
   * For ClassDs, this means superclasses and superinterfaces. Order
   * is depth-first, classes before interfaces, left-to-right, no
   * duplicates. <p/>
   *
   * For TypeVariableDs and WildcardTypeDs, this means bounds. <p/>
   *
   * @return The collection of type declarations for all direct
   * supertypes (including java.lang.Object). <p/>
   *
   * Collection of TypeDs.
   */
  public Collection getHierarchy();

  /**
   * True if this type is assignable from the subject type. Behavior
   * should match reflection. Consider the following expression where
   * S (subject) represents another type and T (this) represents this
   * type. <p/>
   *
   * <pre>
   *   void method( S subject )
   *   {
   *     T variable = subject;
   *   }
   * </pre>
   *
   * Notice that because the parameter subject is not a
   * constant-value, the assignment conversion will not perform the
   * implicit narrowing conversion. <p/>
   *
   * Reflection does not perform auto-boxing and therefore, neither
   * does this. <p/>
   *
   * @return True if the above assignment is allowed. False if the
   * answer is not known or if the answer is no.
   */
  public boolean isAssignableFrom( TypeD subject );


  // ----------------------------------------------------------------------

  /**
   * Gets all declared fields. Includes synthetic fields and enum
   * constants. <p/>
   *
   * @return The collection of field declarations for declared fields,
   * synthetic or not, including enum constants. <p/>
   *
   * Collection of FieldDs.
   */
  public Collection getDeclaredFields();

  /**
   * Gets the first matching declared field, null if none. <p/>
   *
   * @param name The name to match. Require non-null.
   *
   * @return The field declaration for the first matching declared field,
   * null if none.
   */
  public FieldD getDeclaredField( String name );

  /**
   * Gets all declared constructors. Includes synthetic
   * constructors. <p/>
   *
   * Note: Constructors are not inherited, so the collection of
   * declared constructors is the same as the collection of
   * constructors. <p/>
   *
   * @return The collection of constructor declarations for declared
   * constructors, synthetic or not. <p/>
   *
   * Collection of ConstructorDs.
   */
  public Collection getDeclaredConstructors();

  /**
   * Gets the matching declared constructor, null if none. <p/>
   *
   * @param parameters The exact parameter types to match. Null
   * indicates an empty parameter list.
   *
   * @return The constructor declaration for the first matching
   * constructor, null if none.
   */
  public ConstructorD getDeclaredConstructor( TypeD[] parameters );

  /**
   * Gets all declared methods. Does not include the hidden
   * <code>clinit</code> method. Does not include the various "access$"
   * methods that may be present in class files.
   *
   * @return The collection of method declarations for declared methods,
   * excluding the <code>clinit</code> method. <p/>
   *
   * Collection of MethodDs.
   */
  public Collection getDeclaredMethods();

  /**
   * Gets all matching declared methods.
   *
   * @param name The method name to match. Require non-null.
   *
   * @return The collection of method declarations for matching declared
   * methods. <p/>
   *
   * Collection of MethodDs.
   */
  public Collection getDeclaredMethods( String name );

  /**
   * Gets the matching declared method, null if none. <p/>
   *
   * @param name The method name to match. Require non-null.
   *
   * @param parameters The exact parameter types to match. Null
   * indicates an empty parameter list.
   *
   * @return The method declaration for the first matching method, null if none.
   */
  public MethodD getDeclaredMethod( String name, TypeD[] parameters );

  /**
   * Gets all declared member classes. Does not include local and
   * anonymous inner classes. <p/>
   *
   * TODO: Should this return local and anonymous inner classes? I
   * don't remember what reflection does. <p/>
   *
   * @return The collection of class declarations for declared member
   * classes. Only returns member classes that are directly enclosing
   * in this type. Hence, a member class of a member class would not
   * be listed here. <p/>
   *
   * Collection of ClassDs.
   */
  public Collection getDeclaredClasses();

  /**
   * Gets the matching member class, null if none.
   *
   * @return The class declaration for the matching declared member class.
   */
  public ClassD getDeclaredClass( String name );


  // ----------------------------------------------------------------------

  /**
   * Gets all member fields, declared and inherited. The type
   * hierarchy dictates order. Does not filter according to
   * accessibility or visibility. <p/>
   *
   * For more detal on members, please see the Members discussion at
   * {@link TypeD}. <p/>
   *
   * Note: The implicit "length" field for array types should be
   * suppressed because according to the JLS, it is technically not a
   * member variable. <p/>
   *
   * @return A collection of field declarations for all member fields,
   * declared and inherited. <p/>
   *
   * Collection of FieldDs.
   */
  public Collection getFields();

  /**
   * Gets the matching member field, null if none.
   *
   * @param name The field name to match. Require non-null.
   *
   * @return The field declaration for the first matching member field,
   * null if none.
   */
  public FieldD getField( String name );

  /**
   * Gets all member methods, declared and inherited. The type
   * hierarchy dictates order. Does not filter according to
   * accessibility or visibility. <p/>
   *
   * For more detal on members, please see the Members discussion at
   * {@link TypeD}. <p/>
   *
   * @return A collection of method declarations for all member methods,
   * declared and inherited. <p/>
   *
   * Collection of MethodDs.
   */
  public Collection getMethods();

  /**
   * Gets all matching methods.
   *
   * @param name The method name to match. Require non-null.
   *
   * @return A collection of method declarations for all matching member
   * methods. <p/>
   *
   * Collection of MethodDs.
   */
  public Collection getMethods( String name );

  /**
   * Gets the first matching method, null if none.
   *
   * @param name The method name to match. Require non-null.
   * @param parameters The exact parameter types to match. Null
   * indicates an empty parameter list.
   *
   * @return The method declaration for the first matching method, null if
   * none.
   */
  public MethodD getMethod( String name, TypeD[] parameters );

  /**
   * Gets all member classes, declared and inherited. The type
   * hierarchy dictates order.  Does not filter according to
   * accessibility or visibility. <p/>
   *
   * For more detal on members, please see the Members discussion at
   * {@link TypeD}. <p/>
   *
   * @return A collection of class declarations for all member classes,
   * declared and inherited. <p/>
   *
   * Collection of ClassDs.
   */
  public Collection getClasses();

  /**
   * Gets the first matching member class, null if none.
   *
   * @param name The simple class name to match. Require non-null.
   *
   * @return The class declaration for the first matching member class,
   * null if none.
   */
  public ClassD getClass( String name );


  // ----------------------------------------------------------------------

  /**
   * True if the other object is also a TypeD and they represent the
   * same type. <p/>
   *
   * ClassDs and WildcardTypeDs are uniquely identified by their type
   * signatures. TypeVariableDs are uniquely identified by both their
   * owning declaration and type signature. <p/>
   *
   * @return True if o is a TypeD and declarations the same type.
   */
  public boolean equals( Object o );


  // ----------------------------------------------------------------------
}
