/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.util;

import java.net.URL;

import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

/**
 * The description of an icon. An icon is specified either as a key lookup
 * in a resource file which resolves to a resource path relative to the resource
 * file, or an absolute resource path.<p>
 * 
 * Use the {@link #createPathInstance( ClassLoader, String )} or 
 * {@link #createResourceInstance( ClassLoader, String, String)} factory 
 * methods to create an instance of this class. To retrieve the URL of the 
 * icon, use {@link #getURL()}. This URL is normally a resource (i.e. classpath)
 * URL, suitable for use when constructing an <tt>ImageIcon</tt> or other 
 * toolkit specific image type.
 */
public abstract class IconDescription 
{
  private URL _url;

  private IconDescription()
  {
    
  }
  
  /**
   * Create an instance that wraps the specified icon URL.
   * 
   * @param url the url of an icon.
   * @return an IconDescription instance.
   */
  public static IconDescription createInstance( final URL url )
  {
    return new IconDescription()
    {
      protected URL resolveURL() throws MissingIconException
      {
        return url;
      }
    };
  }
  
  /**
   * Create an instance of <tt>IconDescription</tt> based on a resource key
   * lookup.
   * 
   * @param loader the class loader to load the icon resources from.
   * @param bundleClass the class name of the resource bundle.
   * @param key the key of the image path.
   * 
   * @return a new <tt>IconDescription</tt> instance.
   */
  public static IconDescription createResourceInstance( final ClassLoader loader, 
    final String bundleClass, final String key )
  {
    return new IconDescription()
    {
      protected URL resolveURL()
        throws MissingIconException
      {
        
        ResourceBundle bundle;
        try
        {
          bundle = ResourceBundle.getBundle( bundleClass, 
            Locale.getDefault(), loader );
        }
        catch ( MissingResourceException mre )
        {
          throw new MissingIconException( 
            "Resource bundle " + bundleClass + " not found.", 
            mre
          );
        }
        
        try
        {
          String path = bundle.getString( key );
          if ( path == null || path.trim().length() == 0 )
          {
            throw new MissingIconException(
              "Icon key "+key+" has empty value in bundle "+bundleClass
            );
          }
          Class bundleClass = bundle.getClass();
          URL result = bundleClass.getResource( path );
          if ( result == null )
          {
            throw new MissingIconException(
              "Icon path "+path+" does not exist. Specified by key "+key+
              " in bundle "+bundleClass
            );
          }
          return result;
        }
        catch ( MissingResourceException mre )
        {
          throw new MissingIconException( 
            "Icon key "+key+" not defined in bundle "+ bundleClass, 
            mre
          );
        }
      }
    };
  }
  
  /**
   * Create an instance of <tt>IconDescription</tt> based on a direct icon
   * path in the manifest file.
   * 
   * @param loader the class loader to load the icon resource from.
   * @param path the path of the icon.
   * @return a new <tt>IconDescription</tt> instance.
   */
  public static IconDescription createPathInstance( final ClassLoader loader, 
    final String path )
  {
    return new IconDescription()
    {
      protected URL resolveURL()
        throws MissingIconException
      {
        URL resource = loader.getResource( path );
        if ( resource == null )
        {
          throw new MissingIconException(
            "Icon path " + path + " not found."
          );
        }
        return resource;
      }
    };
  }
  
  /**
   * Resolve the URL for this instance of IconDescription.
   * 
   * @return the resolved URL.
   */
  protected abstract URL resolveURL() throws MissingIconException;
  
  /**
   * Get the resolved URL of the icon.
   * 
   * @return the URL of the icon.
   * @throws MissingIconException if the icon could not be loaded.
   */
  public URL getURL()
    throws MissingIconException
  {
    if ( _url == null )
    {
      _url = resolveURL();
    }
    return _url;
  }
}
